#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <string.h>
#include <signal.h>
#include <termios.h>
#define _POSIX_SOURCE 1 

#define SOCK_CMD "/var/run/wavecom_mplex_cmd"
#define SOCK_DATA "/var/run/wavecom_mplex_data"

#define BUF_SIZE 4096

int running;
int serial;
int mux;
int debug;

#define MUX_CMD      0xAA
#define MUX_DATA     0xDD
#define MUX_CMD_TYPE 0x1D
#define HEADER_SIZE  3
#define MAX_SIZE     2047

enum DATA_TYPES { DATA = 0, STATUS, READY, BUSSY };

struct mux_header_t
{
	__uint8_t start;
	__uint8_t lsb;
	__uint8_t msb:3;
	__uint8_t ptype:5;
};

struct data_status_t {
	__uint8_t spare:3;
	__uint8_t ri:1;
	__uint8_t brk:1;
	__uint8_t x:1;
	__uint8_t sb:1;
	__uint8_t sa:1;
};

void usage(char *cmd)
{
	printf("syntax: %s <serial device>\n\n"\
		"example: %s /dev/ttyS1\n"\
		"\n"\
		"this is mplexd a serial multiplexer for wavecom modems\n\n"\
		"THIS IS -> NOT <- WRITTEN BY WAVECOM!!!\n\n"\
		"wavecom modes have a multiplexing mode to send two streams over one "\
		"serial line, one stream for commands (AT) and one for data (GPRS) "\
		"the multiplexer will spawn two sockets one for each stream, the data "\
		"socket will be dead until a GPRS connection is initiated\n\n"\
		"command socket is: %s\n"\
		"data socket is:    %s\n\n"\
		"this is part of the h63xx port Linux port!\n\n"\
		"author is: Collin R. Mulliner <collin(AT)betaversion.net>\n"\
		"version %s\n"
		"", cmd, cmd, SOCK_CMD, SOCK_DATA, VERSION);
}

/**
 *  @brief write n bytes to socket
 *
 *  this is taken from: unix network programming by W.R. Stevens
 *  
 *  @param fd   file descriptor
 *  @param vptr bytes to write out
 *  @param n    number of bytes to write
 *
 *  @returns -1 on failure "n" on success
 */
ssize_t nwrite(int fd, const void *vptr, size_t n)
{
	size_t nleft;
	ssize_t nwritten;
	const char *ptr;
	
	
	ptr = vptr;
	nleft = n;
	while (nleft > 0) {
		if ((nwritten = write(fd, ptr, nleft)) <= 0) {
			if (errno == EINTR) {
				nwritten = 0;
			}
			else {
				return(-1);
			}
		}
		nleft -= nwritten;
		ptr += nwritten;
	}
	return(n);
}

/**
 *  @brief read N bytes from file descriptor
 *
 *  @param fd   file descriptor
 *  @param vptr read buffer
 *  @param n    number of bytes to read
 *
 *  @returns number of bytes read or -1 on error
 */
ssize_t cread(int fd, void *vptr, size_t n)
{
	int result;
	
	
reread:
	if ((result = read(fd, vptr, n)) < 0) {
		if (result == EINTR) {
			result = 0;
			goto reread;
		}
		else {
			return(-1);
		}
	}
	return(result);
}

int get_line(__uint8_t *buf, int length)
{
	int i;
	
	for (i = 0; i < length; i++) {
		if (buf[i] == '\r' || buf[i] == '\n') return(i+1);
	}
	return(0);
}

int calc_checksum(__uint8_t *buf, int length)
{
	int check_sum = 0;
	int i;
	
	for (i = 0; i < length; i++) {
		check_sum += buf[i];
	}
	return(check_sum%256);
}

void sighand(int sig)
{
	running = 0;
	close(serial);
	
	unlink(SOCK_CMD);
	unlink(SOCK_DATA);
}

int main(int argc, char **argv)
{
	struct pollfd fds[3];
	int data = -1;
	int cmd = -1;
	int data_s;
	int cmd_s;
	struct sockaddr_un saddr_data;
	struct sockaddr_un saddr_cmd;	
	unsigned char buf_data[BUF_SIZE];
	unsigned char buf_cmd[BUF_SIZE];
	unsigned char buf_serial[BUF_SIZE];
	unsigned char send_buf[BUF_SIZE];
	int buf_data_p = 0;
	int buf_cmd_p = 0;
	int buf_serial_p = 0;
	int tmp;
	struct mux_header_t *he;
	int serial_pack_len;
	struct data_status_t *ds;
	struct termios newtio;
	int linelen;
	
	
	if (argc < 2) {
		usage(argv[0]);
		exit(1);
	}
	
	running = 1;
	mux = 0;
	debug = 1;
		
	signal(SIGINT, sighand);

	if ((data_s = socket(PF_UNIX, SOCK_STREAM, 0)) < 0) perror("socket(data)");
	if ((cmd_s = socket(PF_UNIX, SOCK_STREAM, 0)) < 0) perror("socket(cmd)");
	strcpy(saddr_data.sun_path, SOCK_DATA);
	strcpy(saddr_cmd.sun_path, SOCK_CMD);
	saddr_cmd.sun_family = AF_UNIX;
	saddr_data.sun_family = AF_UNIX;
	if (bind(data_s, (struct sockaddr*)&saddr_data, sizeof(struct sockaddr_un)) < 0) perror("bind(data)");
	if (bind(cmd_s, (struct sockaddr*)&saddr_cmd, sizeof(struct sockaddr_un)) < 0) perror("bind(cmd)");
	if (listen(cmd_s, 1) < 0) perror("listen(cmd)");
	if (listen(data_s, 1) < 0) perror("listen(data)");


	serial = open(argv[1], O_RDWR|O_NOCTTY);
	if (serial < 0) {
		perror("open serial");
		exit(-1);
	}
	fcntl(serial, F_SETFL, 0);

	// init serial line
	tcgetattr(serial, &newtio);
	cfsetispeed(&newtio, B115200);
	cfsetospeed(&newtio, B115200);
	tcflush(serial, TCIFLUSH);
	tcsetattr(serial, TCSANOW, &newtio);			

	fds[0].fd = serial;
	fds[0].events = 0|POLLIN|POLLPRI;
	fds[1].fd = cmd_s;
	fds[1].events = 0|POLLIN|POLLPRI;
	fds[2].fd = data_s;
	fds[2].events = 0|POLLIN|POLLPRI;
		
	while (poll(fds, 3, -1) >= 0) {
		if (!running) {
			close(serial);
			close(data);
			close(cmd);
			break;
		}
		if ((fds[0].revents & POLLIN) == POLLIN || (fds[0].revents & POLLPRI) == POLLPRI) {
			tmp = cread(serial, buf_serial+buf_serial_p, BUF_SIZE-buf_serial_p);
			if (tmp < 1) {
				if (debug) perror("read(serial)");
				exit(-1);
			}
			else {
				buf_serial_p += tmp;
				if (debug) printf("read(serial)=%d\n", tmp);
			}
			
moreserialdata:
			if (buf_serial_p >= HEADER_SIZE) {
				he = (struct mux_header_t*) buf_serial;
				serial_pack_len = 0 | he->lsb | (he->msb << 8);
				if (debug) printf("length total = %d pack = %d\n", buf_serial_p, serial_pack_len);
				if (buf_serial_p >= serial_pack_len) {
					if (calc_checksum(buf_serial, serial_pack_len+HEADER_SIZE) == buf_serial[serial_pack_len+HEADER_SIZE]) {
						switch (he->start) {
						case MUX_CMD:
							if (debug) printf("got CMD packet length=%d\n", serial_pack_len);
							if (debug) printf("CMD type = %x\n", he->ptype);
							
							if (cmd >= 0) nwrite(cmd, buf_serial+HEADER_SIZE, serial_pack_len);
							if (debug) { nwrite(1, buf_serial+HEADER_SIZE, serial_pack_len); }
							break;
						case MUX_DATA:
							ds = (struct data_status_t*) buf_serial+HEADER_SIZE;
							if (debug) printf("got DATA packet length=%d\n", serial_pack_len);
							if (debug) printf("DATA package type = ");
							switch (he->ptype) {
							case DATA:
								if (debug) printf("DATA\n");
								
								if (data >= 0) nwrite(data, buf_serial+HEADER_SIZE, serial_pack_len);
								if (debug) { nwrite(1, buf_serial+HEADER_SIZE, serial_pack_len); }
								break;
							case STATUS:
								if (debug) printf("STATUS\n");
								if (debug) printf("sa(%d) sb(%d) x(%d) brk(%d) ri(%d)\n", ds->sa, ds->sb, ds->x, ds->brk, ds->ri);
								{
									struct mux_header_t *she = (struct mux_header_t*)send_buf;
									if (debug) printf("got status, send answer status\n");
									she->start = MUX_DATA;
									she->ptype = STATUS;
									she->msb = 0;
									she->lsb = 1;
									send_buf[4] = *(buf_serial+HEADER_SIZE);
									send_buf[HEADER_SIZE+1] = calc_checksum(send_buf, HEADER_SIZE+1);
									if (debug) nwrite(1, send_buf, HEADER_SIZE+2);
									nwrite(serial, send_buf, HEADER_SIZE+2);
								}
								break;
							case READY:
								if (debug) printf("READY\n");
								break;
							case BUSSY:
								if (debug) printf("BUSSY\n");
								break;
							default:
								if (debug) printf("unknown(%d) - ERROR\n", he->ptype);
								break;
							}
							break;
						default:
							break;
						}
clearserialread:
						// clean buffer
						memmove(buf_serial, buf_serial+serial_pack_len+HEADER_SIZE+1, buf_serial_p-serial_pack_len+HEADER_SIZE+1);
						buf_serial_p -= serial_pack_len+HEADER_SIZE+1;
						if (buf_serial_p > 0) goto moreserialdata;
					}
					else {
						if (debug) printf("checksum wrong!\n");
						goto clearserialread;
					}
				}
				else {
					if (debug) printf("serial: got data ... packet not complete ... waiting for more\n");
				}
			}
		}
		if ((fds[1].revents & POLLIN) == POLLIN || (fds[1].revents & POLLPRI) == POLLPRI) {
			if (cmd < 0) {
				if ((cmd = accept(cmd_s, NULL, NULL)) >= 0) {
					fds[1].fd = cmd;
					if (debug) printf("connect(cmd)\n");
				}
			}
			else {
				tmp = cread(cmd, buf_cmd+buf_cmd_p, BUF_SIZE-buf_cmd_p);
				if (tmp < 1) {
					if (debug) perror("read cmd");
					fds[1].fd = cmd_s;
					close(cmd);
					cmd = -1;
					buf_cmd_p = 0;
				}
				else {
					if (debug) printf("read(cmd)=%d\n", tmp);
					buf_cmd_p += tmp;
				}
		
				if ((linelen = get_line(buf_cmd, buf_cmd_p)) > 0) {
					if (linelen > MAX_SIZE) goto clearcmdread;
					struct mux_header_t *she = (struct mux_header_t*)send_buf;
					she->start = MUX_CMD;
					she->ptype = MUX_CMD_TYPE;
					she->msb = linelen >> 8;
					she->lsb = linelen & 0xff;

					if (debug) {
						int i;
						
						for (i = 0; i < linelen; i++) {
							printf("%0.2X", buf_cmd[i]);
						}
						printf("\n");
					}
					
					memcpy(send_buf+HEADER_SIZE, buf_cmd, linelen);
					send_buf[HEADER_SIZE+linelen] = calc_checksum(send_buf, HEADER_SIZE+linelen);

					if (debug) nwrite(1, send_buf, linelen+HEADER_SIZE+1);					
					nwrite(serial, send_buf, linelen+HEADER_SIZE+1);

clearcmdread:
					// clean buffer
					memmove(buf_cmd, buf_cmd+linelen, buf_cmd_p-linelen);
					buf_cmd_p -= linelen;
				}
				else {
					if (debug) printf("cmd line not complete yet\n");
				}
			}
		}
		if ((fds[2].revents & POLLIN) == POLLIN || (fds[2].revents & POLLPRI) == POLLPRI) {
			if (data < 0) {
				if ((data = accept(data_s, NULL, NULL)) >= 0) {
					fds[1].fd = data;
					if (debug) printf("connect(data)\n");
				}
			}
			tmp = cread(data, buf_data+buf_data_p, BUF_SIZE-buf_data_p);
			if (tmp < 1) {
				if (debug) perror("read data");
				fds[2].fd = data_s;
				close(data);
				data = -1;
				buf_data_p = 0;
			}
			else {
				if (debug) printf("read(data)=%d\n", tmp);
				buf_data_p += tmp;
			}
			
			if (buf_data_p > 0) {
				tmp = (buf_data_p > MAX_SIZE) ? MAX_SIZE : buf_data_p;
				struct mux_header_t *she = (struct mux_header_t*)send_buf;
				she->start = MUX_DATA;
				she->ptype = 0;
				she->msb = tmp >> 8;
				she->lsb = tmp & 0xff;
					
				memcpy(send_buf+HEADER_SIZE, buf_data, tmp);
				send_buf[HEADER_SIZE+tmp] = calc_checksum(send_buf, HEADER_SIZE+tmp);
				
				if (debug) nwrite(1, send_buf, tmp + HEADER_SIZE+1);
				nwrite(serial, send_buf, tmp + HEADER_SIZE+1);

				// clean buffer
				memmove(buf_data, buf_data+tmp, buf_data_p-tmp);
				buf_data_p -= tmp;
			}
		}
	}
	
	sighand(-1);
	
	return(1);
}
