#include "fat_fs.h"

int fat_get_next_entry(fat_sb_t *sb, uint32_t curr, uint32_t *val){
        uint32_t num   = curr / sb->entries;
	uint32_t idx   = curr - (sb->entries * num);

        if(curr >= sb->first && curr < sb->last){
                goto out;
        }

        /* Lets try and load the block */
        if(disk_bread((sb->fat_start + num), 1, sb->fat)){
                printf("Failed to read FAT\n");
                return -1;
        }

        sb->first = (num * sb->entries);
        sb->last  = sb->first + 128;
out:
        switch(sb->fat_bits){
        case 32:
                *val = ((uint32_t *) sb->fat)[idx];
		break;
        case 16:
                *val =  (uint32_t) ((uint16_t *) sb->fat)[idx];

		switch(*val){
		case 0xFFF0:
		case 0xFFF6:
		case 0xFFF7:
		case 0xFFF8:
		case 0xFFFF:
			*val |= 0x0FFF0000;
		}
		break;
        case 12:
                /* TODO */
                return -1;
	default:
		// Invalid FAT size
		return -1;
        }

        return 0;
}

size_t build_sting(dir_slot_t *ds, char *name){
	int idx = 0;
	int pos = 0;

	for(idx = 0; idx < 10; idx += 2){
		name[pos++] = ds->name0_4[idx];
		if(ds->name0_4[idx] == '\0'){
			return pos;
		}
	}

	for(idx = 0; idx < 12; idx += 2){
                name[pos++] = ds->name5_10[idx];
                if(ds->name5_10[idx] == '\0'){
                        return pos;
                }
        }

	for(idx = 0; idx < 4; idx += 2){
                name[pos++] = ds->name11_12[idx];
                if(ds->name11_12[idx] == '\0'){
                        return pos;
                }
        }

	name[pos] = '\0';

	return pos;
}

int get_cluster(fat_sb_t *sb, uint32_t *curr){
	uint32_t next 	= 0;
	uint32_t sector = 0;

	// Lets get the next entry in
	// FAT table
	if(fat_get_next_entry(sb,*curr,&next) == -1){
		return 0;
	}

	switch(next){
	case 0x0FFFFFFF:
	case 0x0FFFFFF8:
		// End of file or directory
		*curr = 0x0FFFFFFF;
		return 0;
	case 0x0FFFFFF0:
	case 0x0FFFFFF6:
		// Reserved
		printf("Reserved cluster:%08X\n",next);
		return 0;
	case 0x0FFFFFF7:
	case 0:
		printf("Invalid cluster:%08X\n",next);
		// Bad cluster
		return -1;
	}

	// Calculate the begining of
	// the first sector in the next
	// cluster.
	sector  = sb->data_start + (next * sb->cluster_size);
	
	if(disk_bread(sector, sb->cluster_size, sb->data)){
                return -1;
        }

	// Save number of the
	// new cluster
	*curr = next;

	return 0;
}

dir_entry_t *fat_build_dir_entry(fat_sb_t    *sb, 
				 dir_entry_t *de, 
				 uint8_t     **e,
				 size_t      size, 
				 uint32_t    *curr,
				 char 	     *name){
	char   tmp[256];
	size_t   len = 0;
	uint8_t	*end = *e;

	// Reset the name 
	memset((void *) name, 0, 256);

	do {
		// Cast to directory solt
		dir_slot_t *ds = (dir_slot_t *) de;
		
		// build string
		len += build_sting(ds, tmp);
	
		// Add the new string
		// to the old one and then
		strcat(tmp, name);
		strcpy(name,tmp);

		name[len] = '\0';

		// move to the next one
		++de;
		
		if( (uint8_t *) de >= end){
			if(get_cluster(sb,curr) <= 0){
				return NULL;
			}

			de  = (dir_entry_t *) sb->data;
			end = sb->data + size;
			*e  = end;
		}
	}while(de->attr == ATTR_VFAT);

	return de;
}

int fat_lookup_node(fat_sb_t *sb, char *name, inode_t *node) {
        dir_entry_t *de    = (dir_entry_t *) sb->root;
        size_t      size   = (sb->root_size * sb->sector_size);
        uint8_t      *e    = sb->root + size;
        uint32_t    curr   = 0;

	if(node->block != 0){
		// Load the cuurent directory
		if(disk_bread(node->sector, sb->cluster_size, node->blk)){
                	return -1;
        	}

		de   = (dir_entry_t *) node->blk;
		size = sb->cluster_size * sb->sector_size;
		e    = node->blk + size;
		curr = node->block;
	}
	

	while((de->name[0] != '\0') || (de->attr == ATTR_VFAT)){
        	if(de->attr == ATTR_VFAT){
			// Build long name.
                	de = fat_build_dir_entry(sb,de, &e,size,&curr,node->name);
			if(!de) {
				return -1;
			}

                        if(de->name[0] == 0xe5){
				// deleted file or directory
                        	goto next;
                        }
   		}else if(de->name[0] == 0xe5){
                	// Deleted entry
                        goto next;
               	}else {
			// Short name entry
               		sprintf(node->name, "%.*s.%.*s", 8,de->name,3,de->ext);
               	}

		if(!strcmp(node->name, name)){
			// We have found it.
			node->sb = (void *) sb;
			if(sb->fat_bits == 32){
				// FAT 32
				node->block = ((uint32_t) de->starthi << 16) | (uint32_t) de->start;
			}else {
				// FAT 16
				node->block = de->start;
			}
			node->sector = sb->data_start + (sb->cluster_size * node->block);
			node->size   = de->size;

			return 0;
		}

next:
                ++de;
		if((uint8_t *) de >= e){
                        if(get_cluster(sb,&curr) <= 0){
                                return -1;
                        }

                        de = (dir_entry_t *) sb->data;
                        e  = sb->data + size;
                }
        }

	return -1;
}
